#!/bin/bash

set -eu  # error on command failure, and on unset variables
local_d=$PWD
set +e ; conda deactivate ; set -e

# Some defaults
bench1d=1
bench3d=1
clean=0
parallel='-parallel mpi_omp'
install_d=$local_d


# LMDZ and install_lmdz version to be downloaded
install_svn_version=0
version="-v 20220519.trunk" # Premiere version sous git
version="-v 20221212.trunk"
version="-v 20230114.trunk"
version="-v 20230218.trunk -unstable"
version="-v 20230626.trunk"
version="-v 20241121.trunk -unstable" ; install_svn_version=5341
version="-v 20231022.trunk" ; install_svn_version=4729
version="-v 20240308.trunk" ; install_svn_version=4848
version="-v 20240508.trunk" ; install_svn_version=4848
version="-v 20241018.trunk" ; install_svn_version=5244
version="-v 20250209.trunk -unstable" ; install_svn_version=5528

# Directories management :
# ------------------------
# save_pub="-save_pub -save_pub_dir DIR" given ton install_lmdz.sh
#     Allows saving files donwloaded from lmdz web site
#     Alternative save_pub=""
# netcdf=DIR, -netcdf $netcdf passed to install_lmdz.sh
#     Alternative netcdf=0

LMDZname=LMDZ$( echo $version | sed -e 's/-v//' -e 's/ //g' -e 's/.trunk//' -e 's/unstable//' -e 's/\-//g' )
save_pub_dir="" ; for d in $HOME $local_d ; do if [ -d $d/LMDZ/pub ] ; then save_pub_dir=$d/LMDZ/pub ; fi ; done
save_pub=$( if [ "$save_pub_dir" = "" ] ; then save_pub="" ; else save_pub="-save_pub -save_pub_dir $save_pub_dir" ; fi  )
netcdf=$save_pub_dir/netcdf


#-------------------------------------------------------------------------
function run_cmd_with_log { # Runing a command with log file
#-------------------------------------------------------------------------
echo Runing $*
local logf=$PWD/$( basename $logfile )
echo Log in $logf
$* > $logf 2>&1
echo '      ----        '
if [ $? = 0 ] ; then
   echo Command $1 completed
else
   echo Something went wrong with command $1
   echo Look in log file $logf
   echo and ask for help
   exit 1
fi
echo '      ----        '
}

cat <<eod
-------------------------------------------------------------------------
   -- LMDZ with simple dry 20-parameter physics from FH PHD --
-------------------------------------------------------------------------

Will install the model LMDZ and then add sources from the 20-parameter physics
The model will be installed on
    on $local_d/$LMDZname/modipsl/modeles/LMDZ/
The 20-parameter physics on
    on $local_d/$LMDZname/modipsl/modeles/LMDZ/libf/phyparam
    $save_pub_dir
A 1D test case will be run on
    $local_d/$LMDZname/modipsl/modeles/LMDZ/Bench1D
and a 3D simulation on
    $local_d/$LMDZname/modipsl/modeles/LMDZ/Bench1D

eod
#-------------------------------------------------------------------------
echo 1. INSTALLATION
#-------------------------------------------------------------------------


if [ ! -d $local_d/$LMDZname -o $clean = 1 ] ; then
   cd $install_d
   if [ ! -f install_lmdz.sh ] ; then
      if [ $install_svn_version = 0 ] ; then
          # 1.1 Installation du modele en mode de parallelisation mixte MIPxOpenMP
          wget http://lmdz.lmd.jussieu.fr/pub/install_lmdz.sh -O install_lmdz.sh
      else
          \rm -rf script_install
          svn co -r $install_svn_version http://svn.lmd.jussieu.fr/LMDZ/BOL/script_install
          \cp -f script_install/install_lmdz.sh .
          \rm -rf script_install
      fi
   fi

   chmod +x install_lmdz.sh
   logfile=install_lmdz$$.log
   run_cmd_with_log ./install_lmdz.sh $version $parallel -bench 0 -name $LMDZname -netcdf $netcdf $save_pub

   #----------------------------------------------------------------------
   echo 1.2 Modification du code source pour prendre en compte la physique
   echo a 20 parametres
   #----------------------------------------------------------------------
   cd $LMDZname/modipsl/modeles/LMDZ/libf
   cp -r phydev phyparam
   cp -r dynphy_lonlat/phydev/ dynphy_lonlat/phyparam
   
   #------------------------------------------------------
   # wget plus dispo
   #wget http://www.lmd.jussieu.fr/~hourdin/PARAM/$paramtar
   rsync -av $local_d/src/ ./
   cd phyparam
   ln -s param/* .
   ln -s ../phylmd/iotd* .
   #  A REMETTRE APRES COMMISSION LMDZ ../phylmd/iophys.F90 .
   cd ../..

fi

echo -----------------------------------------------------------------------
echo 2. TEST
echo -----------------------------------------------------------------------

echo -----------------------------------------------------------------------
echo "2.1 Recupération des parametres d'entree"
echo -----------------------------------------------------------------------
cd $install_d/$LMDZname/modipsl/modeles/LMDZ
#wget http://www.lmd.jussieu.fr/~hourdin/PARAM/test_param.tar
mkdir Bench3D
cp $local_d/def_files/* Bench3D/
cp $local_d/utils/* Bench3D/

# 2.2 compilation
if [ -f compile.sh ] ; then mv compile.sh compile_earth.sh ; fi
cat <<eod>| compile.sh
LM=37
IM=32
JM=32
phys=param
dim=3d
if (( \$# > 0 )) ; then dim=\$1 ; fi
case \$dim in 
   1d) resol=\$LM ; para= ; main=scm ; name=scm ;;
   2d) resol=1x\${JM}x\${LM} ; para= ; main=gcm ; name=gcm2d ;;
   3d) resol=\${IM}x\${JM}x\${LM} ; para="$parallel" ; main=gcm ; name=gcm ;;
   *) echo dim \$dim cas non prevu ; exit 1
esac
./makelmdz -d \$resol -p \$phys \$para \$main
eod
cp -r Bench3D Bench1D
chmod +x compile.sh

echo -----------------------------------------------------------------------
echo 2.3 Runing 1D bench
echo -----------------------------------------------------------------------

if [ $bench1d = 1 ] ; then
    echo Compiling the 1D version , logs in
    logfile=out_compile1d.log
    run_cmd_with_log ./compile.sh 1d
    
    cd Bench1D
    sed -i'' -e 's/nday=.*$/nday=60/' run.def
    echo Runing 1D test in :
    echo $PWD
    logfile=out_scm.log
    run_cmd_with_log ../scm.e

    echo ----------------------------------
    echo You may visualize the oputs here :
    echo ----------------------------------
    echo $PWD
    echo by running
    echo ferret -batch tmp.ps -script fig1d.jnl
    cd -
fi
 
echo -----------------------------------------------------------------------
echo 2.4 Runing 3D bench
echo -----------------------------------------------------------------------

if [ $bench3d = 1 ] ; then
    echo Compiling the 3D version , logs in
    logfile=out_compile3d.log
    run_cmd_with_log ./compile.sh 3d
     
    cd Bench3D
    echo Runing 3D test in :
    echo $PWD
    echo logs in :
    logfile=out_gcm.log
    sed -i'' -e 's/nday=.*$/nday=3/' run.def
    if [ "$parallel" = "" ] ; then
       run_cmd_with_log ../gcm.e
    else
       run_cmd_with_log ./run_local.sh 4 1 ../gcm.e
    fi
    echo ----------------------
    echo Outputs available in :
    echo ----------------------
    echo $PWD/phys.nc
fi
