      SUBROUTINE soil(ngrid,nsoil,firstcall,ptherm_i,ptimestep,ptsrf,ptsoil,pcapcal,pfluxgrd,zout)

!=======================================================================
!
!   Auteur:  Frederic Hourdin     30/01/92
!   -------
!
!   objet:  computation of : the soil temperature evolution
!   ------                   the surfacic heat capacity "Capcal"
!                            the surface conduction flux pcapcal
!
!
!   Method: implicit time integration
!   -------
!   Consecutive ground temperatures are related by:
!           T(k) = C(k) + D(k)*T(k-1)  (1)
!   the coefficients C and D are computed at the t-dt time-step.
!   Routine structure:
!   1)new temperatures are computed  using (1)
!   2)C and D coefficients are computed from the new temperature
!     profile for the t+dt time-step
!   3)the coefficients A and B are computed where the diffusive
!     fluxes at the t+dt time-step is given by
!            Fdiff = A + B Ts(t+dt)
!     or     Fdiff = F0 + Capcal (Ts(t+dt)-Ts(t))/dt
!            with F0 = A + B (Ts(t))
!                 Capcal = B*dt
!           
! Conductivité thermique λ (W/mK), Capacité thermique volumétrique ρC (MJ/m³K)
!   Argile/limon sec	0.4	1.5 – 1.6 
! λs  = 1 W m−1 K−1 et rho Cs = 1 × 10^6 J m−3 K−1
! I = sqrt ( lambda rho C )
! z = z' * sqrt( lambda / rho C )
!
!   Interface:
!   ----------
!
!   Arguments:
!   ----------
!   ngird               number of grid-points
!   ptimestep              physical timestep (s)
!   pto(ngrid,nsoil)     temperature at time-step t (K)
!   ptn(ngrid,nsoil)     temperature at time step t+dt (K)
!   pcapcal(ngrid)      specific heat (W*m-2*s*K-1)
!   pfluxgrd(ngrid)      surface diffusive flux from ground (Wm-2)
!   
!=======================================================================
!   declarations:
!   -------------

use soil_ini_mod, only : soil_min_period, soil_dalpha

implicit none
!-----------------------------------------------------------------------
!  arguments
!  ---------

      integer :: ngrid,nsoil
      real :: ptimestep
      real, dimension(ngrid) :: ptsrf,ptherm_i,pcapcal,pfluxgrd
      real, dimension(ngrid,nsoil) :: ptsoil
      real, dimension(nsoil) :: zout
      logical :: firstcall

      integer :: iflag_srf, iflag_impl


!-----------------------------------------------------------------------
!  local arrays
!  ------------

      integer :: ig,jk
      real, dimension(ngrid,nsoil+1) :: zflux
      real, dimension(ngrid) :: z1
      real, dimension(nsoil) :: za

!   local saved variables:
!   ----------------------
      real, save :: lambda
      real, allocatable, save :: zb(:),dz2(:),zc(:,:),zd(:,:)
!$OMP THREADPRIVATE(zb,dz2,zc,zd,lambda)

!-----------------------------------------------------------------------
!   Depthts:
!   --------

      real :: fz,rk,fz1,rk1,rk2
      fz(rk)=fz1*(soil_dalpha**rk-1.)/(soil_dalpha-1.)

      iflag_srf=0

!     print*,'firstcall soil ',firstcall
      if ( firstcall ) then

!-----------------------------------------------------------------------
!   ground levels 
!   grnd=z/l where l is the skin depth of the diurnal cycle:
!   --------------------------------------------------------

!        print*,'nsoil,ngrid,firstcall=',nsoil,ngrid,firstcall
         ALLOCATE(zb(nsoil+1),dz2(nsoil))
         ALLOCATE(zc(ngrid,nsoil),zd(ngrid,nsoil))

!   la premiere couche represente un dixieme de cycle diurne
         fz1=sqrt(soil_min_period/3.14)

         ! dz2 delta_z des couches
         do jk=1,nsoil
            rk1=jk
            rk2=jk-1
            dz2(jk)=fz(rk1)-fz(rk2)
         enddo

         ! zb 1/delta_z entre les demi couches
         do jk=1,nsoil-1
            rk1=jk+.5
            rk2=jk-.5
            zb(jk+1)=1./(fz(rk1)-fz(rk2))
         enddo

         ! dz de la première demi couche
         lambda=fz(.5)*zb(2)
         zb(1)=1./fz(.5) ! pour le noveau schéma implicit
         PRINT*,'full layers, intermediate layers (secoonds)'

         ! Diagnostics
         do jk=1,nsoil
            rk=jk
            rk1=jk+.5
            rk2=jk-.5
            PRINT*,fz(rk1)*fz(rk2)*3.14,fz(rk)*fz(rk)*3.14
         enddo

         print*,'SIZE ',size(zout),nsoil
         do jk=1,nsoil
            print*,' dans soil ',jk
            rk=jk
            zout(jk)=fz(rk)
         enddo

      endif

!=========== Fin des initialisations =================================

  do jk=1,nsoil
     za(jk)=dz2(jk)/ptimestep
  enddo

  iflag_impl=1

!=========== Version explicite =======================================
!  Necessite des pas de temps ultra petits. Sans doute faux

  if ( iflag_impl == 0 ) then

      do ig=1,ngrid
         zflux(ig,1)=ptherm_i(ig)*zb(1)*(ptsoil(ig,1)-ptsrf(ig))
         zflux(ig,nsoil+1)=0.
         pfluxgrd(ig)=zflux(ig,1)
         pcapcal(ig)=ptherm_i(ig)*zb(1)*ptimestep
         Print*,'TSOIL pcapcal',pcapcal(ig)
         print*,'TSOIL zflux',1,zflux(ig,1),za(1)
      enddo

      do jk=2,nsoil
         do ig=1,ngrid
            zflux(ig,jk)=zb(jk)*(ptsoil(ig,jk)-ptsoil(ig,jk-1))
            print*,'TSOIL zflux',jk,zflux(ig,jk),za(jk)
         enddo
      enddo

      do jk=1,nsoil
         do ig=1,ngrid
            ptsoil(ig,jk)=ptsoil(ig,jk)+(zflux(ig,jk+1)-zflux(ig,jk))*ptimestep/za(jk)
            print*,'TSOIL ptsoil',jk,ptsoil(ig,jk)
         enddo
      enddo
  
  else

!=========== Version implicite =======================================
!  Necessite des pas de temps ultra petits. Sans doute faux

      if ( .not. firstcall ) then
!-----------------------------------------------------------------------
!   Computation of the soil temperatures using the Cgrd and Dgrd
!  coefficient computed at the previous time-step:
!  -----------------------------------------------

!    surface temperature
         if (iflag_srf==0) then
            do ig=1,ngrid
               ptsoil(ig,1)=(lambda*zc(ig,2)+ptsrf(ig))/(lambda*(1.-zd(ig,2))+1.)
            enddo
         else
            do ig=1,ngrid
               ptsoil(ig,1)=zc(ig,1)+zd(ig,1)*ptsrf(ig)
            enddo
         endif

!   other temperatures
         do jk=2,nsoil
            do ig=1,ngrid
               ptsoil(ig,jk)=zc(ig,jk)+zd(ig,jk)*ptsoil(ig,jk-1)
            enddo
         enddo

!        print*,'T fond ',maxval(abs(ptsoil(:,nsoil)-200.))

      endif

!-----------------------------------------------------------------------
!   Computation of the Cgrd and Dgrd coefficient for the next step:
!   ---------------------------------------------------------------

      do ig=1,ngrid
         z1(ig)=za(nsoil)+zb(nsoil)
         zc(ig,nsoil)=za(nsoil)*ptsoil(ig,nsoil)/z1(ig)
         zd(ig,nsoil)=zb(nsoil)/z1(ig)
      enddo

      ! On descends la boucle en jk=1 mais on peut s'arrêter à
      ! jk=2 pour iflag_srf=0
      do jk=nsoil-1,1,-1
         do ig=1,ngrid
            z1(ig)=1./(za(jk)+zb(jk)+zb(jk+1)*(1.-zd(ig,jk+1)))
            zc(ig,jk)=(ptsoil(ig,jk)*za(jk)+zb(jk+1)*zc(ig,jk+1))*z1(ig)
            zd(ig,jk)=zb(jk)*z1(ig)
         enddo
      enddo

!-----------------------------------------------------------------------
!   computation of the surface diffusive flux from ground and
!   calorific capacity of the ground:
!   ---------------------------------

      do ig=1,ngrid
         pfluxgrd(ig)=ptherm_i(ig)*zb(2)*(zc(ig,2)+(zd(ig,2)-1.)*ptsoil(ig,1))
         pcapcal(ig)=ptherm_i(ig)*(dz2(1)+ptimestep*(1.-zd(ig,2))*zb(2))
         z1(ig)=lambda*(1.-zd(ig,2))+1.
         pcapcal(ig)=pcapcal(ig)/z1(ig)
         ! Ecriture avec za :
         !pcapcal(ig)=ptherm_i(ig)*ptimestep*(za(1)+1.-zd(ig,2))*zb(2)) / ( lambda*(1-zd(ig,2)) + 1. )
         pfluxgrd(ig)=pfluxgrd(ig)+pcapcal(ig)*(ptsoil(ig,1)*z1(ig)-lambda*zc(ig,2)-ptsrf(ig))/ptimestep
      enddo

  endif

return
end
