import os
import numpy as np
import matplotlib.pyplot as plt
from netCDF4 import Dataset

# Ouverture du fichier
file="phys4.nc"
ncdata=Dataset(file)

# Recuperation des indices de la longitude et de la latitude à tracer
# On va chercher le point le plus proche sur la grille
# En 1D, on pren d le point du 1D quelques soient les valeurs de longitude et latitude
longitude=0.
latitude=10.

# Jour à partir duquel on trace les cycles diurne
day0=2.
nday=3000
daystep=24

lat=np.array(ncdata.variables['lat'][:])
lon=np.array(ncdata.variables['lon'][:])
lat_i=[ilat for ilat,lati in enumerate(lat) if abs(lati-latitude) == np.min(abs(lat-latitude))][0]
lon_i=[ilon for ilon,loni in enumerate(lon) if abs(loni-longitude) == np.min(abs(lon-longitude))][0]

# Lecture des temperatures et altitude
temp=np.array(ncdata.variables['temp'][:,:,lat_i,lon_i]) # [time,z,y,x]
theta=np.array(ncdata.variables['theta'][:,:,lat_i,lon_i]) # [time,z,y,x]
alt=ncdata.variables['geop'][0,:,lat_i,lon_i]/9800
#alt=ncdata.variables['plev'][0,:,lat_i,lon_i]/100.

# Lecture des flux
swtopdn=np.array(ncdata.variables['swtopdn'][:,lat_i,lon_i])
swtopup=np.array(ncdata.variables['swtopup'][:,lat_i,lon_i])
lwtop=np.array(ncdata.variables['lwtop'][:,lat_i,lon_i])
swtop=swtopdn-swtopup
tottop=swtop-lwtop

# wth et Altitudes au niveaux intermédiaires
wth=np.array(ncdata.variables['w_th'][:,1:,lat_i,lon_i])
altw=0.5*(alt[0:len(alt)-1]+alt[1:])

# Les 8 premiers niveaux pour la temperature et la profondeur du sol
tsoil=ncdata.variables['tsoil'][:,0:14,lat_i,lon_i] # [time,z,y,x]
if len(lat) == 1:
    depth=-ncdata.variables['zsoil'][0,0:14,lat_i,lon_i]/1000.
else :
    depth=-ncdata.variables['tsoil'][0,0:14,0,0]/1000.


# Axe des temps en jours
day=np.array([ float(i)/float(daystep) for i in range(len(temp[:,0])) ])

# Lissage en temps et anomalies diurnes
sbx = np.ones(daystep)/daystep
if np.max(day) <= 3. :
   daily_temp=[np.mean(temp,axis=0)]*len(day)
   daily_tsoil=[np.mean(tsoil,axis=0)]*len(day)
else:
   daily_temp=np.apply_along_axis(lambda m: np.convolve(m, sbx, mode='same'), axis=0, arr=temp)
   daily_tsoil=np.apply_along_axis(lambda m: np.convolve(m, sbx, mode='same'), axis=0, arr=tsoil)

anom_a=temp-daily_temp
anom_s=tsoil-daily_tsoil


# Lissage des flux
swtopup_s=np.convolve(swtopup, sbx, mode='same')
swtopdn_s=np.convolve(swtopdn, sbx, mode='same')
lwtop_s=np.convolve(lwtop, sbx, mode='same')
swtop_s=np.convolve(swtop, sbx, mode='same')
tottop_s=np.convolve(tottop, sbx, mode='same')

#-----------------------------------------------------------------------------------------
def t_plot() :
#-----------------------------------------------------------------------------------------
    it0_s=int(daystep/2)
    lw_s=0.7
    lw_=0.7
    jmax=200.
    if np.max(day) > 3. :
       time_=day
    else :
       time_=24.*day

    plt.margins(x=0.)

    # Graph du haut, Flux solaire
    plt.subplot(4,1,2)
    plt.ylim([0.,1300.])
    plt.xlim([0.,jmax])
    yticks=[ 300 * k for k in range(4) ]
    plt.yticks(yticks)
    plt.ylabel("Flux (W/m"+r"$^2$"+")")
    plt.plot(time_,swtop,c="gold",lw=lw_,label='Rayonnement solaire reçu')
    plt.legend()

    # Graph du bas flux et temperature
    plt.subplot(2,1,2)
    plt.xlim([0.,jmax])
    plt.ylim([100.,350.])
    plt.ylabel("Flux (W/m"+r"$^2$"+") et T (K)")
    plt.xlabel("Jour depuis le début de la simulation")
    # Moyennes journalières
    plt.plot(time_,lwtop,c="red",lw=lw_,label="Flux thermique vers l'espace")
    plt.plot(time_,tsoil[:,0],c='green',lw=lw_,label="Temperature de surface")
    plt.plot(time_[12:],daily_tsoil[12:,0],c='green',lw=3)
    plt.plot(time_[12:],    lwtop_s[12:],c="red",lw=3)
    plt.plot(time_[12:],    swtop_s[12:],c="gold",lw=3)

    # Pdf
    plt.margins(x=0.)
    plt.legend()
    pdffile='fig-vol3-chp-gcm-evol0.pdf'
    plt.savefig('tmp.pdf', bbox_inches='tight')
    os.system('pdfcrop tmp.pdf ; \mv -f tmp-crop.pdf '+pdffile+' ; evince '+pdffile)




#-----------------------------------------------------------------------------
# Tracer des cartes t-z
#-----------------------------------------------------------------------------

def prof_atm_sol(itime,alt_,vart_,tsoil_,color,label):
     plt.plot(vart_[itime,:]-273.15,alt,color=color,label=label)
     plt.plot(tsoil_[itime,:]-273.15,depth,color=color)

def zprofs_plot(ncname,varname,Tmin,Tmax,zmin,zmax) :
     plt.subplot(1,2,1)
     ncdata_=Dataset(ncname+'.nc')
     var=np.array(ncdata_.variables[varname][:,:,lat_i,lon_i]) # [time,z,y,x]
     tsoil_=np.array(ncdata_.variables['tsoil'][:,0:14,lat_i,lon_i]) # [time,z,y,x]
     it0=(nday-1)*daystep
     print("it0=",it0)
     itd=np.int((6*24)/daystep)
     prof_atm_sol(it0+0*itd,alt,var,tsoil_,'darkblue'  ,"00:00")
     prof_atm_sol(it0+1*itd,alt,var,tsoil_,'blue' ,"06:00")
     prof_atm_sol(it0+2*itd,alt,var,tsoil_,'orange',"12:00")
     prof_atm_sol(it0+3*itd,alt,var,tsoil_,'red'   ,"18:00")
     plt.axhline(0.,color="brown",linewidth=2.)
     plt.ylabel('Profondeur (m) | Altitude (km)')
     if varname == 'temp' :
         plt.xlabel('Temperature (°C)')
     else :
         plt.xlabel(r'$\theta$'+' (°C)')
     plt.xlim([Tmin,Tmax])
     plt.ylim([zmin,zmax])
     plt.legend()
     pdffile='fig-vol3-chp-gcm-zprof'+ncname+varname+'.pdf'
     plt.savefig('tmp.pdf')
     os.system('pdfcrop tmp.pdf ; \mv -f tmp-crop.pdf '+pdffile+' ; evince '+pdffile)
     plt.close()
     plt.clf()

def zt_plot() :

     time=(day-day0)*24
     nd=2
     iti=(nday-3)*daystep
     itf=iti+nd*daystep+1
     time=[ k*1. for k in range(nd*daystep+1) ]
     anom_levels=[ float(i-8)/2. for i in range(17) ]
     # Anomalies de températures
     cp=plt.contourf(time,alt,anom_a[iti:itf,:].transpose(),anom_levels,cmap='bwr',extend='both')
     cp=plt.contourf(time,depth,anom_s[iti:itf,:].transpose(),anom_levels,cmap='bwr',extend='both')
     plt.xticks(np.arange(0, nd*daystep+1, 6.0))
     plt.axhline(0.01,color="brown",linewidth=2.)
     print('OK1')
     
     # Tracer des vecteurs
     plt.colorbar(cp) 
     print(np.max(wth[iti:itf,:],axis=0))
     for k in range(len(altw)) :
        plt.plot(time,len(time)*[altw[k]],'+',c='k',ms=0.2)
        q=plt.quiver(time,altw[k],0*wth[iti:itf,k],wth[iti:itf,k],units='height',scale_units='height',angles='uv',scale=200,width=0.004)
        # Tracer de la legende du vent.
        # Il faut choisir une couche avec des vents non nuls
        if k == 3 :
            plt.quiverkey(q, X=0.75, Y=0.9, U=10, label='10 m/s', labelpos='E', coordinates='figure')
     
     for k in range(len(depth)) :
        plt.plot(time,len(time)*[depth[k]],'+',c='k',ms=0.2)
     
     plt.xlabel('Heures')
     plt.ylabel('Altitude (km)')
     plt.xlim([0.,48.])
     plt.ylim([-1.,5.])
     plt.savefig('tmp.pdf')
     pdffile='fig-vol3-chp-gcm-wth.pdf'
     os.system('pdfcrop tmp.pdf ; \mv -f tmp-crop.pdf '+pdffile+' ; evince '+pdffile)

zt_plot()
t_plot()
for ncname_ in ["phys0","phys4"] :
   zprofs_plot(ncname_,'temp' ,10.,60.,-1.,1.)
   zprofs_plot(ncname_,'theta',20.,60.,-1.,6.)
#test_plot() ; quit()
